# importresults.py
# Copyright 2008 Roger Marsh
# Licence: See LICENCE (BSD licence)

"""Extract results from ECF submission files or League database."""

import collections

from chessvalidate.core import gameresults

from . import constants


def get_event_identifier_from_game(game):
    """Return event identifier from game details."""
    return (
        game[constants.STARTDATE],
        game[constants.ENDDATE],
        game[constants.EVENT],
    )


def get_section_identifier_from_game(game):
    """Return section identifier from game details."""
    return (
        game[constants.STARTDATE],
        game[constants.ENDDATE],
        game[constants.EVENT],
        game[constants.SECTIONNAME],
        game[constants.UNIQUESECTION],
    )


def get_player_identifier_from_game(game, player, pin):
    """Return player identifier from game details.

    Produces same key as section case in collation.Player.get_player_identity

    """
    return (
        game[player],
        game[constants.EVENT],
        game[constants.STARTDATE],
        game[constants.ENDDATE],
        game[constants.SECTION],
        game[pin],
    )


class ImportResults:
    """Class for importing results data."""

    def __init__(self, textlines, filename):
        """Initialise for import report of results in textlines."""
        del filename
        super().__init__()
        self.textlines = textlines
        self.game = {}
        self.event = {}
        self.match = {}

    def _translate_results_format(self, context=None, items=None):
        """Extract results into a common format.

        Provide rules in context and items arguments.

        """

        def null(data):
            del data

        if context is None:
            context = {}
        for c in context:
            if not isinstance(context[c], collections.abc.Callable):
                context[c] = null
        if items is None:
            items = {}
        data = {}
        for t in self.get_lines():
            ts = t.split("=", 1)
            key, value = ts[0], ts[-1]
            if key not in items:
                if len(key) != 0:
                    return False
            data[key] = value
            if key in context:
                context[key](data)
                data = {}
        if len(data):
            return False
        return True

    def get_event_names(self):
        """Return sorted list of event names."""
        return sorted(list(self.event))

    def get_lines(self):
        """Return self.textlines.

        Null get_lines method. Define get_lines method in subclass if
        self.textlines needs transforming before being processed by
        translate_results_format method.

        """
        return self.textlines


class ImportMerges(ImportResults):
    """Import data from file generated by merges.py."""

    def translate_results_format(self):
        """Extract result and player identification data."""

        def game(data):
            if (
                data[constants.RESULT] in gameresults.storeresults
            ):  # constants._storeresults:
                gamenumber = len(self.game)
                self.game[gamenumber] = data
                sections = self.event.setdefault(
                    get_event_identifier_from_game(data), set()
                )
                sections.add(data[constants.SECTION])
                uniquesections = self.match.setdefault(
                    get_section_identifier_from_game(data), []
                )
                uniquesections.append(gamenumber)
                for pin in (constants.HOMEPIN, constants.AWAYPIN):
                    data[pin] = pinmap.setdefault(data[pin], len(pinmap) + 1)

        pinmap = {}

        context = {
            constants.RESULT: game,
        }

        items = {
            constants.STARTDATE: True,
            constants.ENDDATE: True,
            constants.EVENT: True,
            constants.SECTION: True,
            constants.DATE: True,
            constants.HOMEPLAYERWHITE: True,
            constants.HOMEPLAYER: True,
            constants.AWAYPLAYER: True,
            constants.HOMEPIN: True,
            constants.AWAYPIN: True,
            constants.HOMEAFFILIATION: True,
            constants.AWAYAFFILIATION: True,
            constants.SECTIONNAME: True,
            constants.UNIQUESECTION: True,
            constants.RESULT: True,
            constants.HOMETEAM: False,
            constants.AWAYTEAM: False,
            constants.BOARD_LOWER: False,
            constants.ROUND_LOWER: False,
        }

        return super()._translate_results_format(context=context, items=items)

    def get_lines(self):
        """Delimiter is newline sequence."""
        return [t.strip() for t in self.textlines]


def get_import_event_results(data, filename):
    """Extract results from data and return extracted results."""
    for Format in (ImportMerges,):
        importdata = Format(data, filename)
        if importdata.translate_results_format():
            return importdata
    return None
