# convertresults.py
# Copyright 2009 Roger Marsh
# Licence: See LICENCE (BSD licence)

"""Convert ECF submission files generated from same database.

This module allows event data held on a League database to be extracted
and prepared for import into a Berkeley DB or DPT results database. The
ECF submission file format is used to transfer the data because it is
the only output generated by any grading program in a known and fit for
purpose format.

The assumption necessary is that a PIN value refers to the same player
across submission files.

A restriction is that use of a PIN that looks like an ECF grading code is
sufficient to prevent this module making the assumption for the player.
(The League program does not use the grading code as PIN.)

BCF CODE; CLUB NAME; CLUB (non-standard for CLUB NAME); CLUB CODE;
and CLUB COUNTY fields on submission files are ignored.

Data compression uses bz2. Pickle is not used, before compression, to make
things a bit easier if the destination is not (otherwise) using Python.

(There is a data dump program for League databases and that format is
supported here. The data dump program is available to graders on request.
The only advantage from using this program is that all the data is in
a single file making it difficult to forget some data.)

"""

import collections

from chessvalidate.core import gameresults
import chessvalidate.core.constants as cvcc

from . import matchteams
from . import constants as cc

# This should be used upstream but could not locate place.
homeplayerwhitemap = {True: "yes", False: "no"}


class ConvertResults:
    """Class for importing results data."""

    def __init__(self, pinprefix):
        """Initialise data structes for converting data to internal format."""
        super().__init__()
        self.pinprefix = pinprefix
        self.converterror = None
        self.player = {}
        self.game = {}
        self.event = {}
        self.match = {}
        self.eventteams = {}

    def cycle_name(self, name):
        """Generate versions of name starting with non-initials."""
        c = [name.split()]
        for e in range(len(c[-1]) - 1):
            n = c[-1][:]
            n.append(n.pop(0))
            c.append(n)
        return [" ".join(e) for e in c if len(e[0]) > 1 and not e[0].isupper()]

    def export_games(self, pins=False):
        """Generate list of game results in <field=value> format.

        Player pins can be used to merge players from different submission
        files if pins == True. Pin can always be used to merge players
        within a submission file (references to a player in different
        sections in an event). It is safe to use pins == True if it can be
        guaranteed that same pin means same player on all the submission
        files being processed.

        """
        items = (
            cc.STARTDATE,
            cc.ENDDATE,
            cc.EVENT,
            cc.SECTION,
            cc.DATE,
            # cc.HOMEPLAYERWHITE,
            cc.HOMEPLAYER,
            cc.AWAYPLAYER,
        )
        if pins:
            pin_items = (
                (cc.HOMEPIN, cc.HOMEPIN),
                (cc.AWAYPIN, cc.AWAYPIN),
            )
        else:
            pin_items = (
                (cc.HOMEPIN, cc.HOMESERIAL),
                (cc.AWAYPIN, cc.AWAYSERIAL),
            )
        optional_items = (
            cc.HOMETEAM,
            cc.AWAYTEAM,
            cc.BOARD_LOWER,
            cc.ROUND_LOWER,
        )

        players = self.player
        matches = self.match
        exportgames = []
        for game in self.game.values():
            homeplayerwhite = homeplayerwhitemap.get(
                game.get(cc.LGCOLOR, game.get(cc.HOMEPLAYERWHITE)), ""
            )
            game[cc.HOMEPLAYERWHITE] = homeplayerwhite
            homeaffiliation = players[
                get_player_identifier_from_game(
                    game, cc.HOMEPLAYER, cc.HOMESERIAL
                )
            ][cc.AFFILIATION]
            awayaffiliation = players[
                get_player_identifier_from_game(
                    game, cc.AWAYPLAYER, cc.AWAYSERIAL
                )
            ][cc.AFFILIATION]
            sectionname = matches[game[cc.LMCODE]][cc.LMNAME]
            uniquesection = game[cc.LMCODE]
            for i in items:
                exportgames.append("=".join((i, game[i])))
            for i, j in pin_items:
                exportgames.append("=".join((i, game[j])))
            for i in optional_items:
                e = game.get(i)
                if e is not None:
                    exportgames.append("=".join((i, e)))
            exportgames.append("=".join((cc.HOMEPLAYERWHITE, homeplayerwhite)))
            exportgames.append("=".join((cc.HOMEAFFILIATION, homeaffiliation)))
            exportgames.append("=".join((cc.AWAYAFFILIATION, awayaffiliation)))
            exportgames.append("=".join((cc.SECTIONNAME, sectionname)))
            exportgames.append("=".join((cc.UNIQUESECTION, uniquesection)))
            exportgames.append("=".join((cc.RESULT, game[cc.RESULT])))

        return exportgames

    def get_team_names_from_match_names(self):
        """Generate possible team names.

        Try to get team names from match name using the MatchTeams class.  If
        that fails use the TeamNames class to do the best possible by splitting
        the match name in two.

        MatchTeams, given enough consistent match names, will get team names
        'Team A' and 'Team B' from match name 'Team A - Team B' but TeamNames
        might get as far as team names 'Team A -' and 'Team B'.

        """
        hometeam = set()
        awayteam = set()
        for match in self.match.values():
            eventteams = self.eventteams.setdefault(match[cc.LECODE], {})
            matchteam = eventteams.setdefault(
                match[cc.LMCODE],
                matchteams.MatchTeams(
                    string=match[cc.LMNAME],
                    split=match[cc.LMTYPE] == cc.SECTION_IS_MATCH,
                ),
            )
            for h, a in matchteam.teamsplits:
                if matchteam.position[h] == 0:
                    hometeam.add(h)
                    awayteam.add(a)
        teams = hometeam.intersection(awayteam)
        for eventteams in self.eventteams.values():
            for key, matchteam in eventteams.items():
                matchteam.teamsplits = tuple(
                    (h, a)
                    for h, a in matchteam.teamsplits
                    if h in teams and a in teams
                )
                try:
                    self.match[key][cc.HOMETEAM] = matchteam.teamsplits[-1][0]
                    self.match[key][cc.AWAYTEAM] = matchteam.teamsplits[-1][-1]
                except:
                    teamnames = TeamNames(matchteam.string, teams)
                    matchteam.teamsplits = (
                        (teamnames(cc.HOMETEAM), teamnames(cc.AWAYTEAM)),
                    )
                    self.match[key][cc.HOMETEAM] = matchteam.teamsplits[-1][0]
                    self.match[key][cc.AWAYTEAM] = matchteam.teamsplits[-1][-1]

    def add_match_section_to_events(self):
        """Generate section name for all matches in every event."""
        # collect all the non-match section names
        for match in self.match.values():
            if match[cc.LMTYPE] != cc.SECTION_IS_MATCH:
                event = self.event[match[cc.LECODE]]
                sections = event.setdefault(cc.SECTIONS, [])
                if match[cc.LMNAME] not in sections:
                    sections.append(match[cc.LMNAME])
                else:
                    self.converterror = (
                        "Section name ",
                        match[cc.LMNAME],
                        " duplicated in ",
                        event[cc.LENAME],
                        " event",
                    )
                    return self.empty_extract()

        # check that the system generated match section name is not used
        for match in self.match.values():
            if match[cc.LMTYPE] == cc.SECTION_IS_MATCH:
                event = self.event[match[cc.LECODE]]
                sections = event.get(cc.SECTIONS, [])
                if cc.EVENT_MATCHES in sections:
                    self.converterror = (
                        "Section name for matches (",
                        cc.EVENT_MATCHES,
                        ") is already in ",
                        event[cc.LENAME],
                        " event. Change that section name to proceed.",
                    )
                    return self.empty_extract()

        # add the system generated match section to all events with matches
        for match in self.match.values():
            if match[cc.LMTYPE] == cc.SECTION_IS_MATCH:
                event = self.event[match[cc.LECODE]]
                sections = event.setdefault(cc.SECTIONS, [])
                if cc.EVENT_MATCHES not in sections:
                    sections.append(cc.EVENT_MATCHES)
        self.get_team_names_from_match_names()
        return True

    def change_game_player_identifiers(self, game):
        """Change player id from <event, player> to <player, event, section>.

        one entry per player per event existed
        but need one entry per player per section per event
        copy player details from game to player for both players

        """
        match = self.match[game[cc.LMCODE]]
        if match[cc.LMTYPE] == cc.SECTION_IS_MATCH:
            eventsection = cc.EVENT_MATCHES
        else:
            eventsection = match[cc.LMNAME]
        for gplayer, gserial, side, pin in (
            (cc.HOMEPLAYER, cc.HOMESERIAL, 0, cc.HOMEPIN),
            (cc.AWAYPLAYER, cc.AWAYSERIAL, -1, cc.AWAYPIN),
        ):
            player = self.player.setdefault(
                get_player_identifier_from_game(game, gplayer, gserial), {}
            )
            self.set_player_detail(player, game, gplayer, pin)
            # add game details to team and affiliation derivation data
            self.set_potential_names(
                player,
                side,
                self.eventteams[match[cc.LECODE]][game[cc.LMCODE]],
                eventsection,
            )

    def copy_event_and_player_identifiers_to_games(self):
        """Copy identifiers to games."""
        # copy event and player identifiers to game
        for game in self.game.values():
            match = self.match[game[cc.LMCODE]]
            event = self.event[match[cc.LECODE]]
            for a in cc.EVENT, cc.STARTDATE, cc.ENDDATE:
                game[a] = event[a]
            s = match[cc.LMNAME]
            is_match = match[cc.LMTYPE] == cc.SECTION_IS_MATCH
            if is_match:
                game[cc.SECTION] = cc.EVENT_MATCHES
            else:
                game[cc.SECTION] = s
            c = game.get(cc.LGCOLOR, None)
            if c == cc.WHITE:
                game[cc.HOMEPLAYERWHITE] = cc.YES
            elif c == cc.BLACK:
                game[cc.HOMEPLAYERWHITE] = cc.NO
            else:
                game[cc.HOMEPLAYERWHITE] = cc.NOCOLOR
            for player, pin, serial, pcode in (
                (cc.HOMEPLAYER, cc.HOMEPIN, cc.HOMESERIAL, cc.LPCODE1),
                (cc.AWAYPLAYER, cc.AWAYPIN, cc.AWAYSERIAL, cc.LPCODE2),
            ):
                game[player] = self.player[game[pcode]][cc.PLAYER]
                game[pin] = self.player[game[pcode]][cc.LPCODE]
                game[serial] = self.player[game[pcode]][cc.SERIAL]
        return True

    def empty_extract(self):
        """Clear data structures."""
        self.player.clear()
        self.game.clear()
        self.event.clear()
        self.match.clear()
        self.eventteams.clear()
        return False

    def _translate_results_format(
        self,
        context=None,
        keymap=None,
        validmap=None,
        pinreadmap=None,
        pinmap=None,
        gradingcodemap=None,
    ):
        """Extract results into a common format.

        Provide rules in context and keymap arguments.

        """

        def null(data, context):
            """Do nothing."""  # Avoid pylint W0613 if 'pass' instead.
            del data, context

        process = null

        if context is None:
            context = {}
        for c in context:
            if not isinstance(context[c], collections.abc.Callable):
                context[c] = null
        if keymap is None:
            keymap = {}
        if validmap is None:
            validmap = {}
        if pinreadmap is None:
            pinreadmap = set()
        if pinmap is None:
            pinmap = set()
        if gradingcodemap is None:
            gradingcodemap = set()

        pinvaluemap = {}
        data = {}
        # Subclasses provide the get_lines method.
        # pylint: disable-next=no-member.
        for t in self.get_lines():
            ts = t.split("=", 1)
            key, value = ts[0], ts[-1]
            if key not in validmap:
                if len(key) != 0:
                    self.converterror = ("Keyword not expected : ", key)
                    return self.empty_extract()
            else:
                vm = validmap[key]
                if isinstance(vm, str):
                    if contextkey != vm:
                        self.converterror = (
                            "Keyword ",
                            key,
                            " not expected after keyword ",
                            contextkey,
                        )
                        return self.empty_extract()
                elif isinstance(vm, dict):
                    try:
                        if contextkey not in vm:
                            self.converterror = (
                                "Keyword ",
                                key,
                                " not expected after keyword ",
                                contextkey,
                            )
                            return self.empty_extract()
                    except:
                        self.converterror = (
                            "Keyword ",
                            key,
                            " not expected before context determined",
                        )
                        return self.empty_extract()
                elif vm is not None:
                    self.converterror = (
                        "Unable to determine validity of keyword ",
                        key,
                    )
                    return self.empty_extract()
            if key in context:
                if len(data):
                    process(data, contextkey)
                process = context[key]
                data = {}
                contextkey = key
            if key in keymap:
                if key in pinmap:
                    if value not in pinvaluemap:
                        if len(value) != cc.GRADING_CODE_LENGTH:
                            pinvaluemap[value] = value
                        elif (
                            value[-1] in cc.GRADING_CODE_CHECK_CHARACTERS
                            and value[:-1].isdigit()
                        ):
                            pinvaluemap[value] = "-".join(
                                (self.pinprefix, str(len(pinvaluemap)))
                            )
                        else:
                            pinvaluemap[value] = value
                if key in pinreadmap:
                    data[keymap[key]] = pinvaluemap[value]
                else:
                    data[keymap[key]] = value
            elif key in gradingcodemap:
                if cc.LPCODE in data:
                    if len(value) == cc.GRADING_CODE_LENGTH:
                        if value[:-1] in data[cc.LPCODE]:
                            self.converterror = (
                                "Grading code ",
                                value,
                                " is included in player pin ",
                                data[cc.LPCODE],
                            )
                            return self.empty_extract()

        if len(data):
            process(data, contextkey)

        self.prune(self.match, self.game, cc.LMCODE)
        self.prune(self.event, self.match, cc.LECODE)
        return True

    def get_game_players(self):
        """Return the set of players who have played games."""
        gplayers = set()
        codes = (cc.LPCODE1, cc.LPCODE2)
        for g in self.game.values():
            for pc in codes:
                if pc in g:
                    gplayers.add(g[pc])
        return gplayers

    def prune(self, target, source, attr):
        """Remove objects from target not referenced by source using attr."""
        used = set()
        for v in source.values():
            if attr in v:
                if v[attr] in target:
                    used.add(v[attr])
        for p in list(target.keys()):
            if p not in used:
                del target[p]

    def prune_players(self, gameplayers):
        """Remove players who have not played any games."""
        for k in list(self.player.keys()):
            if k not in gameplayers:
                del self.player[k]

    def rebuild_player_data(self):
        """Change player identity to <player, event, section>."""
        # copy event and player identifiers to game
        self.player.clear()
        for game in self.game.values():
            self.change_game_player_identifiers(game)

    def report_games(self):
        """Report all games and games for players with affiliation doubts.

        All games sorted by event and section.
        Games for players with affiliation doubts sorted by player.

        """
        gamesbysection = {}
        problemplayers = {}

        for k, game in self.game.items():
            section = gamesbysection.setdefault(
                (
                    game[cc.STARTDATE],
                    game[cc.ENDDATE],
                    game[cc.EVENT],
                    game[cc.SECTION],
                    game[cc.LMCODE],
                ),
                [],
            )
            section.append(k)
            if self.match[game[cc.LMCODE]][cc.LMTYPE] != cc.SECTION_IS_MATCH:
                continue
            for teamplayer, teamserial, team in (
                (cc.HOMEPLAYER, cc.HOMESERIAL, cc.HOMETEAM),
                (cc.AWAYPLAYER, cc.AWAYSERIAL, cc.AWAYTEAM),
            ):
                p = get_player_identifier_from_game(
                    game, teamplayer, teamserial
                )
                if self.player[p][cc.AFFILIATION] not in game[team]:
                    pp = problemplayers.setdefault(p, set())
                    pp.add(k)

        lines = []
        for gbs in sorted(gamesbysection):
            home = away = ""
            sectiongames = []
            for g in gamesbysection[gbs]:
                game = self.game[g]
                if not home:
                    if cc.HOMETEAM in game:
                        home = game[cc.HOMETEAM]
                if not away:
                    if cc.AWAYTEAM in game:
                        away = game[cc.AWAYTEAM]
                sectiongames.append(
                    "".join(
                        (
                            "\t\t",
                            game[cc.HOMEPLAYER],
                            "\t\t\t",
                            self.player[
                                get_player_identifier_from_game(
                                    game, cc.HOMEPLAYER, cc.HOMESERIAL
                                )
                            ][cc.AFFILIATION],
                            "\t\t\t",
                            gameresults.storeresults[game[cc.RESULT]],
                            "\t\t",
                            game[cc.AWAYPLAYER],
                            "\t\t\t",
                            self.player[
                                get_player_identifier_from_game(
                                    game, cc.AWAYPLAYER, cc.AWAYSERIAL
                                )
                            ][cc.AFFILIATION],
                            "\t\t\t",
                            game[cc.DATE],
                        )
                    )
                )
            lines.append(
                "\n".join(
                    (
                        "\t\t".join(gbs[:-1]),
                        "\t\t".join(("\t\t\t", home, "\t\t\t\t", away)),
                        "\n".join(sectiongames),
                    )
                )
            )

        gamesreport = "\n\n".join(lines)

        lines = []
        for name, pp in sorted(
            [
                (self.player[p][cc.PLAYER], p)
                for p in list(problemplayers.keys())
            ]
        ):
            playergames = []
            player = "".join(
                (
                    name,
                    "\t\t\t",
                    self.player[pp][cc.AFFILIATION],
                    "\t\t\t",
                    self.player[pp][cc.STARTDATE],
                    " ",
                    self.player[pp][cc.ENDDATE],
                    "\t\t\t",
                    self.player[pp][cc.EVENT],
                )
            )
            for pg in problemplayers[pp]:
                game = self.game[pg]
                playergames.append(
                    "".join(
                        (
                            "\t\t\t",
                            game[cc.DATE],
                            "\t\t\t",
                            game[cc.HOMEPLAYER],
                            "\t\t\t",
                            game[cc.HOMETEAM],
                            "\t\t\t",
                            game[cc.AWAYTEAM],
                            "\t\t\t",
                            game[cc.AWAYPLAYER],
                        )
                    )
                )
            lines.append("\n".join((player, "\n".join(playergames))))

        playergamesreport = "\n\n".join(lines)

        playeraffiliations = []
        for k, v in self.player.items():
            startdate, enddate, event, player, section, serial = k
            affiliation = v[cc.AFFILIATION]
            for p in self.cycle_name(player):
                playeraffiliations.append(
                    (
                        p,
                        affiliation,
                        startdate,
                        enddate,
                        event,
                        section,
                        serial,
                    )
                )
        playeraffiliations.sort()

        playeraffiliationreport = "\n".join(
            [
                "\t\t".join((p, "\t", a, "\t", sd, ed, e, "\t", s))
                for p, a, sd, ed, e, s, sn in playeraffiliations
            ]
        )

        events = []
        for e in self.event.values():
            events.append(
                (
                    e[cc.STARTDATE],
                    e[cc.ENDDATE],
                    e[cc.EVENT],
                    e[cc.SECTIONS],
                )
            )
        events.sort()

        eventreport = "\n".join(
            [
                "".join(
                    (
                        sd,
                        "\t\t",
                        ed,
                        "\t\t",
                        e,
                        "\n\t\t\t\t",
                        "\n\t\t\t\t".join(list(sections)),
                        "\n",
                    )
                )
                for sd, ed, e, sections in events
            ]
        )

        return (
            eventreport,
            playeraffiliationreport,
            playergamesreport,
            gamesreport,
        )

    def set_player_detail(self, player, game, gplayer, gpin):
        """Copy name and pin detail from game to player."""
        if not player:
            player[cc.PLAYER] = game[gplayer]
            player[cc.PIN_LOWER] = game[gpin]
            for a in cc.EVENT, cc.SECTION, cc.STARTDATE, cc.ENDDATE:
                player[a] = game[a]

    def set_potential_names(self, player, side, eventteams, section):
        """Add potential team names and affiliations to player[cc.NAMES]."""
        names = player.setdefault(cc.NAMES, {})
        names[section] = names.setdefault(section, {})
        phrases = {}
        for ts in eventteams.teamsplits:
            phrases[ts[side]] = phrases.setdefault(ts[side], 0) + 1
        for p in phrases:
            counter = names[section].setdefault(p, PhraseCounts(p))
            if side == 0:
                counter += 1
            else:
                counter += 1

    def set_team_names_and_affiliations(self):
        """Derive player's affiliation from sections (matches) played."""
        teams = set()
        for eps, player in self.player.items():
            # pick most common phrase as player affiliation
            # if it is not the match name it is a potential team name
            for phrases in player[cc.NAMES].values():
                phrase = sorted(
                    [(c.get_weight(), c) for c in phrases.values()],
                    reverse=True,
                )[0][-1]
                player[cc.AFFILIATION] = phrase.chars
                if not phrase.equal:
                    teams.add(phrase.chars)
                # remove if not finished with deleted names
                player[cc.NAMES] = [c for c in phrases.values() if c.equal]

        # pick team names that are best fit to match name as affiliations
        for eps, player in self.player.items():
            if eps[-1] != cc.EVENT_MATCHES:
                continue
            # affiliation is a match name or a team name
            # find best match between affiliation and potential team names
            # if there is a best match make that the affiliation
            # if not (it is a tie) set affiliation to the choices
            affiliation = player[cc.AFFILIATION]
            if affiliation not in player[cc.NAMES]:
                if affiliation in teams:
                    # affiliation is a team so nothing needs doing
                    continue
            # find most common home or away team name in affiliation
            w = affiliation.split()
            ateam = None
            hteam = None
            for i in range(1, len(w) + 1):
                t = " ".join(w[:i])
                if t in teams:
                    hteam = t
                t = " ".join(w[-i:])
                if t in teams:
                    ateam = t
            tcounts = {}
            for phrase in player[cc.NAMES]:
                for t in ateam, hteam:
                    if t:
                        if t in phrase.chars:
                            tcounts[t] = tcounts.setdefault(t, 0) + 1
            # adjust the affiliation
            if tcounts.get(ateam, 0) > tcounts.get(hteam, 0):
                player[cc.AFFILIATION] = ateam
            elif tcounts.get(hteam, 0) > tcounts.get(ateam, 0):
                player[cc.AFFILIATION] = hteam
            else:
                player[cc.AFFILIATION] = (hteam, ateam)

        # set team names for game
        match = self.match
        for game in self.game.values():
            if match[game[cc.LMCODE]][cc.LMTYPE] != cc.SECTION_IS_MATCH:
                continue
            for team in cc.HOMETEAM, cc.AWAYTEAM:
                game[team] = match[game[cc.LMCODE]][team]

        # where player affiliation has not been resolved to a team name use
        # the last suitable game team name encountered as affiliation but
        # give priority to a name in the affiliation tuple
        for game in self.game.values():
            if self.match[game[cc.LMCODE]][cc.LMTYPE] != cc.SECTION_IS_MATCH:
                continue
            # set player affiliation same as game team
            for teamplayer, team, serial in (
                (cc.HOMEPLAYER, cc.HOMETEAM, cc.HOMESERIAL),
                (cc.AWAYPLAYER, cc.AWAYTEAM, cc.AWAYSERIAL),
            ):
                player = self.player[
                    get_player_identifier_from_game(game, teamplayer, serial)
                ]
                if isinstance(player[cc.AFFILIATION], str):
                    # affiliation already set
                    continue
                player[cc.AFFILIATION] = game[team]


class ConvertSubmissionFile(ConvertResults):
    """Import data from file formatted as ECF results submission file."""

    results = {
        cvcc.RESULT_01: cvcc.AWIN,  # cc._loss,
        cvcc.RESULT_55: cvcc.DRAW,  # cc._draw,
        cvcc.RESULT_10: cvcc.HWIN,  # cc._win,
    }
    colour = {
        cc.ECF_COLOUR_WHITE: True,  # cc.WHITE,
        cc.ECF_COLOUR_BLACK: False,  # cc.BLACK,
        cc.ECF_COLOUR_W: True,  # cc.WHITE,
        cc.ECF_COLOUR_B: False,  # cc.BLACK,
    }
    colourdefault = {
        cc.ECF_COLOURDEFAULT_ALL: cc.WHITE_ON_ALL,
        cc.ECF_COLOURDEFAULT_EVEN: cc.BLACK_ON_ODD,
        cc.ECF_COLOURDEFAULT_NONE: cc.BLACK_ON_ALL,
        cc.ECF_COLOURDEFAULT_ODD: cc.WHITE_ON_ODD,
        cc.ECF_COLOURDEFAULT_UNKNOWN: cc.COLOR_NOT_SPECIFIED,
    }

    def translate_results_format(self):
        """Translate results to internal format."""

        def convert_colour_text(data):
            # try:
            # data[cc.LGCOLOR] = ConvertSubmissionFile.colour[
            # data[cc.LGCOLOR].lower()]
            # except:
            # data[cc.LGCOLOR] = cc.NOCOLOR
            try:
                data[cc.LGCOLOR] = ConvertSubmissionFile.colour[
                    data[cc.LGCOLOR].lower()
                ]
            except:
                data[cc.LGCOLOR] = None  # cc.NOCOLOR

        def convert_colour_default_text(data):
            try:
                data[cc.LMCOLOR] = ConvertSubmissionFile.colourdefault[
                    data[cc.LMCOLOR].lower()
                ]
            except:
                data[cc.LMCOLOR] = cc.NOCOLOR

        def convert_result_text(data):
            try:
                data[cc.LGRESULT] = ConvertSubmissionFile.results[
                    data[cc.LGRESULT]
                ]
            except:
                data[cc.LGRESULT] = cc.VOID

        def get_event(data, context):
            del context
            k = str(len(self.event) + 1)
            self.event[k] = data
            if cc.LECODE in data:
                data[cc.LECODE] = k
            convert_date_to_iso(data, cc.LEDATE)
            convert_date_to_iso(data, cc.LEFINALDATE)

        def get_game(data, context):
            del context
            convert_result_text(data)
            if (
                data[cc.LGRESULT] in gameresults.storeresults
            ):  # cc._storeresults:
                k = str(len(self.game) + 1)
                self.game[k] = data
                data[cc.LGCODE] = k
                data[cc.LMCODE] = str(len(self.match))
                e = str(len(self.event))
                for p in (cc.LPCODE1, cc.LPCODE2):
                    data[p] = (e, data[p])
                convert_date_to_iso(data, cc.LGDATE)
                convert_colour_text(data)

        def get_match(data, context):
            k = str(len(self.match) + 1)
            self.match[k] = data
            data[cc.LMCODE] = k
            data[cc.LECODE] = str(len(self.event))
            if context in sectiontypemap:
                data[cc.LMTYPE] = sectiontypemap[context]
            convert_date_to_iso(data, cc.LMDATE)
            convert_colour_default_text(data)

        def get_player(data, context):
            del context
            self.player[(str(len(self.event)), data[cc.LPCODE])] = data
            data[cc.SERIAL] = str(len(self.player))
            if cc.LCNAME in data:
                if data[cc.LCNAME][-1] == "*":
                    data[cc.LCNAME] = data[cc.LCNAME][:-1].strip()
            if cc.LPNAME not in data:
                if cc.E_SURNAME in data:
                    data[cc.LPNAME] = " ".join(
                        (
                            "".join((data.get(cc.E_SURNAME, ""), ",")),
                            " ".join(
                                (
                                    data.get(cc.E_FORENAME, ""),
                                    data.get(cc.E_INITIALS, ""),
                                )
                            ).strip(),
                        )
                    ).strip()

        context = {
            cc.EVENT_DETAILS: get_event,
            cc.PLAYER_LIST: get_player,
            cc.OTHER_RESULTS: get_match,
            cc.MATCH_RESULTS: get_match,
            cc.SECTION_RESULTS: get_match,
            cc.FINISH: None,
            cc.PIN: get_player,
            cc.PIN1: get_game,
        }

        keymap = {
            cc.EVENT_CODE: cc.LECODE,
            cc.EVENT_NAME: cc.LENAME,
            cc.EVENT_DATE: cc.LEDATE,
            cc.FINAL_RESULT_DATE: cc.LEFINALDATE,
            cc.PIN: cc.LPCODE,
            cc.NAME: cc.LPNAME,
            cc.OTHER_RESULTS: cc.LMNAME,
            cc.MATCH_RESULTS: cc.LMNAME,
            cc.SECTION_RESULTS: cc.LMNAME,
            cc.RESULTS_DATE: cc.LMDATE,
            cc.PIN1: cc.LPCODE1,
            cc.PIN2: cc.LPCODE2,
            cc.ROUND: cc.LGROUND,
            cc.BOARD: cc.LGBOARD,
            cc.COLOUR: cc.LGCOLOR,
            cc.SCORE: cc.LGRESULT,
            cc.GAME_DATE: cc.LGDATE,
            cc.WHITE_ON: cc.LMCOLOR,
            # cc.CLUB:cc.LCNAME, #League program for CLUB NAME
            # cc.CLUB_NAME:cc.LCNAME,
            cc.SURNAME: cc.E_SURNAME,
            cc.INITIALS: cc.E_INITIALS,
            cc.FORENAME: cc.E_FORENAME,
        }

        sectiontypemap = {
            cc.OTHER_RESULTS: cc.OTHER_TYPE,
            cc.MATCH_RESULTS: cc.LEAGUE_MATCH_TYPE,
            cc.SECTION_RESULTS: cc.TOURNAMENT_TYPE,
        }

        # validmap rejects at least one field that is mandatory on a valid
        # submission file, SUBMISSION_INDEX, and at least one set from which
        # one will be present on a valid submission file, BCF_CODE CLUB_NAME
        # or CLUB, for each player.
        # validmap should be identical to version in preparesource.py with
        # relevant entries commented out.
        validmap = {
            # cc.ADJUDICATED:cc.EVENT_DETAILS,
            # cc.BCF_CODE:cc.PIN,
            # cc.BCF_NO:cc.PIN,
            cc.BOARD: {
                cc.MATCH_RESULTS: None,
                cc.PIN1: None,
            },
            # cc.CLUB:cc.PIN, #League program for CLUB NAME
            # cc.CLUB_CODE:cc.PIN,
            # cc.CLUB_COUNTY:cc.PIN,
            # cc.CLUB_NAME:cc.PIN,
            cc.COLOUR: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
                cc.OTHER_RESULTS: None,
                cc.PIN1: None,
            },
            # cc.COMMENT:{
            # cc.PIN:None,
            # cc.MATCH_RESULTS:None,
            # cc.SECTION_RESULTS:None,
            # },
            # cc.DATE_OF_BIRTH:cc.PIN,
            cc.EVENT_CODE: cc.EVENT_DETAILS,
            cc.EVENT_DATE: cc.EVENT_DETAILS,
            cc.EVENT_DETAILS: None,
            cc.EVENT_NAME: cc.EVENT_DETAILS,
            # cc.FIDE_NO:cc.PIN,
            cc.FINAL_RESULT_DATE: cc.EVENT_DETAILS,
            cc.FINISH: None,
            cc.FORENAME: cc.PIN,
            cc.GAME_DATE: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
                cc.OTHER_RESULTS: None,
                cc.PIN1: None,
            },
            # cc.GENDER:cc.PIN,
            # cc.INFORM_CHESSMOVES:cc.EVENT_DETAILS,
            # cc.INFORM_FIDE:cc.EVENT_DETAILS,
            # cc.INFORM_GRAND_PRIX:cc.EVENT_DETAILS,
            # cc.INFORM_UNION:cc.EVENT_DETAILS,
            cc.INITIALS: cc.PIN,
            cc.MATCH_RESULTS: None,
            # cc.MINUTES_FIRST_SESSION:cc.EVENT_DETAILS,
            # cc.MINUTES_FOR_GAME:cc.EVENT_DETAILS,
            # cc.MINUTES_REST_OF_GAME:cc.EVENT_DETAILS,
            # cc.MINUTES_SECOND_SESSION:cc.EVENT_DETAILS,
            # cc.MOVES_FIRST_SESSION:cc.EVENT_DETAILS,
            # cc.MOVES_SECOND_SESSION:cc.EVENT_DETAILS,
            cc.NAME: cc.PIN,
            cc.OTHER_RESULTS: None,
            cc.PIN: {
                cc.PLAYER_LIST: None,
                cc.PIN: None,
            },
            cc.PIN1: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
                cc.OTHER_RESULTS: None,
                cc.PIN1: None,
            },
            cc.PIN2: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
                cc.OTHER_RESULTS: None,
                cc.PIN1: None,
            },
            cc.PLAYER_LIST: None,
            cc.RESULTS_DATE: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
            },
            # cc.RESULTS_DUPLICATED:cc.EVENT_DETAILS,
            # cc.RESULTS_OFFICER:cc.EVENT_DETAILS,
            # cc.RESULTS_OFFICER_ADDRESS:cc.EVENT_DETAILS,
            cc.ROUND: {
                cc.SECTION_RESULTS: None,
                cc.PIN1: None,
            },
            cc.SCORE: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
                cc.OTHER_RESULTS: None,
                cc.PIN1: None,
            },
            # cc.SECONDS_PER_MOVE:cc.EVENT_DETAILS,
            cc.SECTION_RESULTS: None,
            # cc.SUBMISSION_INDEX:cc.EVENT_DETAILS,
            cc.SURNAME: cc.PIN,
            # cc.TITLE:cc.PIN,
            # cc.TREASURER:cc.EVENT_DETAILS,
            # cc.TREASURER_ADDRESS:cc.EVENT_DETAILS,
            cc.WHITE_ON: {
                cc.MATCH_RESULTS: None,
                cc.SECTION_RESULTS: None,
                cc.OTHER_RESULTS: None,
            },
        }

        extract = super()._translate_results_format(
            context=context,
            keymap=keymap,
            validmap=validmap,
            pinreadmap={cc.PIN, cc.PIN1, cc.PIN2},
            pinmap={cc.PIN},
            gradingcodemap={cc.BCF_CODE},
        )

        if not extract:
            return False

        self.prune_players(self.get_game_players())

        if not self.add_match_section_to_events():
            return False

        if not self.copy_event_and_player_identifiers_to_games():
            return False

        self.rebuild_player_data()

        self.set_team_names_and_affiliations()

        return extract


class ConvertLeagueDump(ConvertResults):
    """Import data from dump of League program database."""

    results = {
        cc.RESULT_0: cc.TBR,
        cc.RESULT_1: cvcc.AWIN,  # cc._loss,
        cc.RESULT_2: cvcc.DRAW,  # cc._draw,
        cc.RESULT_3: cvcc.HWIN,  # cc._win,
        cc.RESULT_4: cc.AWAYDEFAULT,
        cc.RESULT_5: cc.HOMEDEFAULT,
        cc.RESULT_6: cc.VOID,
        cc.RESULT_7: cc.WINBYE,
        cc.RESULT_8: cc.DRAWBYE,
    }
    colour = {
        cc.COLOUR_1: True,  # cc.WHITE,
        cc.COLOUR_2: False,  # cc.BLACK,
        cc.COLOUR_0: None,  # cc.NOCOLOR,
    }
    colourdefault = {
        cc.COLOURDEFAULT_1: cc.WHITE_ON_ALL,
        cc.COLOURDEFAULT_4: cc.BLACK_ON_ODD,
        cc.COLOURDEFAULT_0: cc.COLOR_NOT_SPECIFIED,
        cc.COLOURDEFAULT_2: cc.WHITE_ON_ODD,
        cc.COLOURDEFAULT_3: cc.BLACK_ON_ALL,
    }

    def __init__(self, pinprefix):
        """Initialise data structures."""
        super().__init__(pinprefix=pinprefix)
        self.affiliate = {}
        self.club = {}
        self.represent = set()
        self.team = {}

    def empty_extract(self):
        """Clear data structures."""
        self.affiliate.clear()
        self.club.clear()
        self.represent.clear()
        self.team.clear()
        return super().empty_extract()

    def translate_match_field(self):
        """Translate matches to internal format."""
        fixturelines = []
        reportlines = []
        matchresults = {}
        # Method translate_match_field is not called.
        # pylint: disable-next=no-member.
        for tf in sorted(self.files):
            mname = ""
            with open(tf, "r", encoding="utf8") as infi:  # 'r' or 'rb'?
                for tt in infi:
                    ts = tt.split("=", 1)
                    f = ts[0]
                    if f == cc.LMNAME:
                        mname = ts[-1]
                    elif f == cc.LMTYPE:
                        if ts[-1].rstrip() == cc.SECTION_IS_MATCH:
                            mn = "_".join(
                                (cc.TAKEON_MATCH, str(len(matchresults) + 1))
                            )
                            fixturelines.append("=".join((mn, mname.rstrip())))
                            reportlines.append("=".join((f, mn.rstrip())))
                            matchresults[mn] = (
                                len(fixturelines),
                                len(reportlines),
                            )
                        else:
                            reportlines.append(tt.rstrip())
                    else:
                        reportlines.append(tt.rstrip())
        return (fixturelines, reportlines, matchresults)

    def translate_results_format(self):
        """Translate results to internal format."""
        # pylint W0612 unused-variable.
        # context dict has commented 'affiliate' key.
        # def get_affiliate(data, context):
        #     del context
        #     self.affiliate[(data[cc.LECODE], data[cc.LPCODE])] = data
        #     if cc.LPNAME in data:
        #         del data[cc.LPNAME]
        #     if cc.LEDATE in data:
        #         del data[cc.LEDATE]

        # pylint W0612 unused-variable.
        # context dict has commented 'club' key.
        # def get_club(data, context):
        #     del context
        #     self.club[data[cc.LCCODE]] = data
        #     if cc.LCNAME in data:
        #         if data[cc.LCNAME][-1] == "*":
        #             data[cc.LCNAME] = data[cc.LCNAME][:-1].strip()

        def convert_colour_text(data):
            try:
                data[cc.LGCOLOR] = ConvertLeagueDump.colour[data[cc.LGCOLOR]]
            except:
                data[cc.LGCOLOR] = cc.NOCOLOR

        def convert_colour_default_text(data):
            try:
                data[cc.LMCOLOR] = ConvertLeagueDump.colourdefault[
                    data[cc.LMCOLOR].lower()
                ]
            except:
                data[cc.LMCOLOR] = cc.NOCOLOR

        def convert_result_text(data):
            try:
                data[cc.LGRESULT] = ConvertLeagueDump.results[
                    data[cc.LGRESULT]
                ]
            except:
                data[cc.LGRESULT] = cc.VOID

        def get_event(data, context):
            del context
            self.event[data[cc.LECODE]] = data
            convert_date_to_iso(data, cc.LEDATE)
            convert_date_to_iso(data, cc.LEFINALDATE)

        def get_game(data, context):
            del context
            convert_result_text(data)
            if (
                data[cc.LGRESULT] in gameresults.storeresults
            ):  # cc._storeresults:
                self.game[data[cc.LGCODE]] = data
                convert_date_to_iso(data, cc.LGDATE)
                convert_colour_text(data)

        def get_match(data, context):
            del context
            self.match[data[cc.LMCODE]] = data
            convert_date_to_iso(data, cc.LMDATE)
            convert_colour_default_text(data)

        def get_player(data, context):
            del context
            self.player[data[cc.LPCODE]] = data
            data[cc.SERIAL] = str(len(self.player))
            lnn = int(data[cc.LPLENNICKNAME])
            lfn = int(data[cc.LPLENFORENAME])
            n = data[cc.LPNAME]
            if lnn and lfn:
                # pycodestyle E203 whitespace before ':'.
                # black formatting insists on the space.
                data[cc.LPNAME] = " ".join(
                    ("".join((n[: -lfn - lnn - 4], ",")), n[-lfn - lnn - 3 :])
                )
            elif lfn:
                data[cc.LPNAME] = " ".join(
                    ("".join((n[: -lfn - 1], ",")), n[-lfn:])
                )
            elif lnn:
                # pycodestyle E203 whitespace before ':'.
                # black formatting insists on the space.
                data[cc.LPNAME] = " ".join(
                    ("".join((n[: -lnn - 3], ",")), n[-lnn - 2 :])
                )

        # pylint W0612 unused-variable.
        # context dict has commented 'represent' key.
        # def get_represent(data, context):
        #     del context
        #     self.represent.add((data[cc.LTCODE], data[cc.LPCODE]))

        def get_team(data, context):
            del context
            self.team[data[cc.LTCODE]] = data

        context = {
            # cc.LREPRESENT:get_represent,
            # cc.LCLUB:None,
            cc.LPLAYER: get_player,
            cc.LGAME: get_game,
            # cc.LAFFILIATE:None,
            cc.LTEAM: get_team,
            cc.LEVENT: get_event,
            cc.LMATCH: get_match,
        }

        keymap = {
            cc.ECODE: cc.LECODE,
            cc.ENAME: cc.LENAME,
            cc.EDATE: cc.LEDATE,
            cc.EFINALDATE: cc.LEFINALDATE,
            cc.PCODE: cc.LPCODE,
            cc.PNAME: cc.LPNAME,
            cc.MCODE: cc.LMCODE,
            cc.MNAME: cc.LMNAME,
            cc.MDATE: cc.LMDATE,
            cc.PCODE1: cc.LPCODE1,
            cc.PCODE2: cc.LPCODE2,
            cc.GCODE: cc.LGCODE,
            cc.GROUND: cc.LGROUND,
            cc.GBOARD: cc.LGBOARD,
            cc.GCOLOR: cc.LGCOLOR,
            cc.GRESULT: cc.LGRESULT,
            cc.GDATE: cc.LGDATE,
            cc.MCOLOR: cc.LMCOLOR,
            cc.MTYPE: cc.LMTYPE,
            # cc.CCODE:cc.LCCODE,
            # cc.CNAME:cc.LCNAME,
            cc.TCODE: cc.LTCODE,
            cc.TNAME: cc.LTNAME,
            # cc.RPAIRING:cc.LRPAIRING,
            cc.TCODE1: cc.LTCODE1,
            cc.TCODE2: cc.LTCODE2,
            cc.PLENFORENAME: cc.LPLENFORENAME,
            cc.PLENNICKNAME: cc.LPLENNICKNAME,
        }

        validmap = {
            cc.ECODE: {cc.LEVENT: None, cc.LMATCH: None, cc.LAFFILIATE: None},
            cc.ENAME: cc.LEVENT,
            # cc.EBCF:cc.LEVENT,
            cc.EDATE: {cc.LEVENT: None, cc.LAFFILIATE: None},
            cc.EFINALDATE: cc.LEVENT,
            # cc.ESUBMISSION:cc.LEVENT,
            # cc.ETREASURER:cc.LEVENT,
            # cc.EADDRESS1:cc.LEVENT,
            # cc.EADDRESS2:cc.LEVENT,
            # cc.EADDRESS3:cc.LEVENT,
            # cc.EADDRESS4:cc.LEVENT,
            # cc.EPOSTCODE:cc.LEVENT,
            # cc.EGRADER:cc.LEVENT,
            # cc.EGADDRESS1:cc.LEVENT,
            # cc.EGADDRESS2:cc.LEVENT,
            # cc.EGADDRESS3:cc.LEVENT,
            # cc.EGADDRESS4:cc.LEVENT,
            # cc.EGPOSTCODE:cc.LEVENT,
            # cc.EFIRSTMOVES:cc.LEVENT,
            # cc.EFIRSTMINUTES:cc.LEVENT,
            # cc.ENEXTMOVES:cc.LEVENT,
            # cc.ENEXTMINUTES:cc.LEVENT,
            # cc.ERESTMINUTES:cc.LEVENT,
            # cc.EALLMINUTES:cc.LEVENT,
            # cc.ESECPERMOVE:cc.LEVENT,
            # cc.EADJUDICATED:cc.LEVENT,
            # cc.EGRANDPRIX:cc.LEVENT,
            # cc.EFIDE:cc.LEVENT,
            # cc.ECHESSMOVES:cc.LEVENT,
            # cc.EEAST:cc.LEVENT,
            # cc.EMIDLAND:cc.LEVENT,
            # cc.ENORTH:cc.LEVENT,
            # cc.ESOUTH:cc.LEVENT,
            # cc.EWEST:cc.LEVENT,
            # cc.ECOLOR:cc.LEVENT,
            # cc.CCODE:{cc.LCLUB:None, cc.LTEAM:None, cc.LAFFILIATE:None},
            # cc.CNAME:cc.LCLUB,
            # cc.CBCF:cc.LCLUB,
            # cc.CBCFCOUNTY:cc.LCLUB,
            cc.PCODE: {
                cc.LPLAYER: None,
                cc.LAFFILIATE: None,
                cc.LREPRESENT: None,
            },
            cc.PNAME: {
                cc.LPLAYER: None,
                cc.LAFFILIATE: None,
                cc.LREPRESENT: None,
            },
            # cc.PBCF:cc.LPLAYER,
            # cc.PDOB:cc.LPLAYER,
            # cc.PGENDER:cc.LPLAYER,
            # cc.PDIRECT:cc.LPLAYER,
            # cc.PTITLE:cc.LPLAYER,
            # cc.PFIDE:cc.LPLAYER,
            cc.PLENFORENAME: cc.LPLAYER,
            cc.PLENNICKNAME: cc.LPLAYER,
            cc.MCODE: {cc.LMATCH: None, cc.LGAME: None},
            cc.MNAME: cc.LMATCH,
            cc.MDATE: cc.LMATCH,
            cc.MTYPE: cc.LMATCH,
            cc.MCOLOR: cc.LMATCH,
            # cc.MUSEEVENTDATE:cc.LMATCH,
            cc.TCODE1: cc.LMATCH,
            cc.TCODE2: cc.LMATCH,
            cc.GROUND: cc.LGAME,
            cc.GBOARD: cc.LGAME,
            cc.GCODE: cc.LGAME,
            cc.PCODE1: cc.LGAME,
            cc.PCODE2: cc.LGAME,
            cc.GCOLOR: cc.LGAME,
            cc.GRESULT: cc.LGAME,
            cc.GDATE: cc.LGAME,
            # cc.GUSEMATCHDATE:cc.LGAME,
            cc.TCODE: {cc.LTEAM: None, cc.LREPRESENT: None},
            cc.TNAME: cc.LTEAM,
            # cc.RPAIRING:cc.LREPRESENT,
            # cc.LREPRESENT:None,
            # cc.LCLUB:None,
            cc.LPLAYER: None,
            cc.LGAME: None,
            # cc.LAFFILIATE:None,
            cc.LTEAM: None,
            cc.LEVENT: None,
            cc.LMATCH: None,
        }

        extract = super()._translate_results_format(
            context=context,
            keymap=keymap,
            validmap=validmap,
            pinreadmap={cc.PCODE, cc.PCODE1, cc.PCODE2},
            pinmap={cc.PCODE1, cc.PCODE2},
        )

        if not extract:
            return False

        gplayers = self.get_game_players()
        self.prune_players(gplayers)
        self.prune_affiliations(gplayers)
        del gplayers

        if not self.add_match_section_to_events():
            return False
        if not self.copy_event_and_player_identifiers_to_games():
            return False

        self.rebuild_player_data()
        self.set_team_names_and_affiliations()
        return extract

    def prune_affiliations(self, gameplayers):
        """Remove affiliation for players who have not played any games."""
        for e, p in list(self.affiliate.keys()):
            if p not in gameplayers:
                del self.affiliate[(e, p)]


class PhraseCounts:
    """Counts of phrase usage.

    The number of times the phrase equals starts or ends strings is updated
    by Add method. Phrases that start or end a string are candidate team
    names. Phrases that equal a string are likely match names.

    """

    def __init__(self, phrase):
        """Initialise counts which contribute to phrase weight."""
        super().__init__()
        self.phrase = tuple(phrase.split())
        self.chars = phrase
        self.equal = 0
        self.count = 0

    def __add__(self, increment):
        """Add increment to self.count contribution to phrase weight."""
        self.count += increment

    def get_weight(self):
        """Return weight assigned to phrase."""
        return (self.count, len(self.phrase))

    def __eq__(self, other):
        """Return True if defining attributea are equal."""
        if self.count != other.count:
            return False
        if len(self.phrase) != len(other.phrase):
            return False
        if self.chars != other.chars:
            return False
        return True

    def __ne__(self, other):
        """Return True if any defining attribute is not equal."""
        if self.count == other.count:
            return False
        if len(self.phrase) == len(other.phrase):
            return False
        if self.chars == other.chars:
            return False
        return True

    def __ge__(self, other):
        """Return True if defining attributes in order are greater or equal."""
        if self.count < other.count:
            return False
        if len(self.phrase) < len(other.phrase):
            return False
        if self.chars < other.chars:
            return False
        return True

    def __gt__(self, other):
        """Return True if defining attributes in order are greater."""
        if self.count <= other.count:
            return False
        if len(self.phrase) <= len(other.phrase):
            return False
        if self.chars <= other.chars:
            return False
        return True

    def __le__(self, other):
        """Return True if defining attributes in order are smaller or equal."""
        if self.count > other.count:
            return False
        if len(self.phrase) > len(other.phrase):
            return False
        if self.chars > other.chars:
            return False
        return True

    def __lt__(self, other):
        """Return True if defining attributes in order are smaller or equal."""
        if self.count >= other.count:
            return False
        if len(self.phrase) >= len(other.phrase):
            return False
        if self.chars >= other.chars:
            return False
        return True


def convert_date_to_iso(data, key):
    """Convert assumed dd/mm/yyyy to yyyy-mm-dd."""
    try:
        d = data[key].split("/")
        d.reverse()
        data[key] = "-".join(d)
    except:
        pass


def get_player_identifier_from_game(game, player, serial):
    """Return player identifier from game details."""
    return (
        game[cc.STARTDATE],
        game[cc.ENDDATE],
        game[cc.EVENT],
        game[player],
        game[cc.SECTION],
        game[serial],
    )


def single_splits(words=None):
    """Generate list of single splits of words excluding empty sequence.

    words can be a sequence of strings or a string to to be split
    into words delimited by whitespace

    single_splits('a b c') returns [('a', 'b c'), ('a b', 'c')]

    """
    if words is None:
        return set()
    if isinstance(words, str):
        words = words.split()
    return [
        (" ".join(words[:i]), " ".join(words[i:]))
        for i in range(1, len(words))
    ]


class TeamNames(list):
    """Default team names for a match."""

    def __init__(self, matchname, teams):
        """Extend and deduce home and away team names."""
        super().__init__()
        self.append({cc.HOMETEAM: "", cc.AWAYTEAM: ""})
        self.matchname = matchname

        for h, a in single_splits(words=matchname):
            f = {cc.HOMETEAM: "", cc.AWAYTEAM: ""}
            for t in teams:
                if t in a:
                    if len(t) > len(f[cc.AWAYTEAM]):
                        f[cc.AWAYTEAM] = t
                if t in h:
                    if len(t) > len(f[cc.HOMETEAM]):
                        f[cc.HOMETEAM] = t
            if not f[cc.AWAYTEAM]:
                break
            if f[cc.HOMETEAM]:
                self.append(f)

        if len(self) == 1:
            s = matchname.split()
            self[:] = [
                {
                    cc.HOMETEAM: " ".join(s[: (1 + len(s)) // 2]),
                    cc.AWAYTEAM: " ".join(s[(1 + len(s)) // 2 :]),
                }
            ]
        else:
            del self[:-1]

    def __call__(self, team):
        """Return home or away team."""
        return self[-1][team]
